/***************************************************************** 
*  @file CRYS_HOST_TST_ECC.c
*  @brief Functions used for HOST-SEP communication to send message from HOST to SEP
*         that contains requested API to be executed in SEP and API parameters
*         Used for SEP External application ATP testing.
*
*  @author mottig
*  @remarks Copyright (C) 2004 by Discretix Technologies Ltd.
*           All Rights reserved
*/



/********************** Include Files *************************/

/* .............. CRYS level includes ................. */

#include "DX_VOS_Mem.h"
#include "DX_VOS_Memmap.h"
#include "DX_VOS_Sem.h"
#include "SEPDriver.h" 
#include "ExtApp_qa_op_code.h"
#include "gen.h"
#include "error.h"
#include "CRYS_HOST_TST_RSA.h"
#include "ExtApp_CRYS_Defines.h"


/************************ Defines ******************************/
/************************ MACROS *******************************/
/************************ Global Data **************************/
/**************** Private function prototype *******************/




/************************ Public Functions ******************************/

 
/***********************************************************************
* Function Name:                                                           
*  SEPQA_ECC_ECDSA_NonIntegratedSignVerify         
* 
* Inputs:
* KeyGenType - key gen type (generate internally or build keys)
* DomainID - domain id
* PublKeyIn_ptr   - The pointer to external public key data.
* PublKeySizeInBytes -	Size of public key data in bytes (according to EC domain),
*						it should be equal to 2*modulusSize (CRYS_ECPKI_DomainID_WMDRM10) or 
*						2*modulusSize + 1byte (other domains).
* PrivKeyIn_ptr -      Pointer to private key data. 
* PrivKeySizeInBytes - Size of private key data in bytes. Must be great than null and
*					   less or equall to EC OrderSizeInBytes.
* DataIn_ptr - input data
* DataInSize - input data size
* SignatureSize - size of the sugnature
* HashMode - hash mode
* SignatureSize - the size of signature buffer 
*				  must be not less than 2*OrderSizeInBytes.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                              returns the name of the function that failures
* FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
*  This function is used to operate the qa external application on the SEP, 
*  that executes Acceptance test for non integrated ECC ESDSA Sign & Verify 
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with all non buffer parameters of the API
* 3. send all buffers params                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
*
**************************************************************************/                                
EXPORT_C DxUint32 SEPQA_ECC_ECDSA_NonIntegratedSignVerify(
                            	DxUint32_t			      KeyGenType,
                            	CRYS_ECPKI_DomainID_t     DomainID,
                            	DxUint8_t			     *PublKeyIn_ptr,       /*in*/									
                            	DxUint32_t                PublKeySizeInBytes,  /*in*/
                            	DxUint8_t			     *PrivKeyIn_ptr,     /*in*/
                            	DxUint32_t                PrivKeySizeInBytes,/*in*/
                            	CRYS_ECPKI_HASH_OpMode_t  HashMode,
                            	DxUint8_t				 *DataIn_ptr,
                            	DxUint32_t                DataInSize,
                            	DxUint32_t                SignatureSize,
                            	char                     *FailureFunctionNameOut_ptr,
                                DxUint32_t                FuncNameMaxSize)
{
   /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  
   
   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = 7;

   /* parameters */
   DxUint32_t   Host_messageParam[7] = {0};
   
   /* op code */
   DxUint32_t   Host_opCode = 0;
   	 
  
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   

   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_ECC_ECDSA_NonIntegratedSignVerify;

   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
   	Host_messageParam[DX_SEP_HOST_ECC_DATASIZE_IN_OFFSET]   = DataInSize;
    Host_messageParam[DX_SEP_HOST_ECC_SIGNATURESIZE_OFFSET] = SignatureSize;
    Host_messageParam[DX_SEP_HOST_ECC_KEYGENTYPE_OFFSET]    = KeyGenType;
    Host_messageParam[DX_SEP_HOST_ECC_DOMAINID_OFFSET]      = DomainID;        
    Host_messageParam[DX_SEP_HOST_ECC_PUBLICKEYSIZE_OFFSET] = PublKeySizeInBytes;        
    Host_messageParam[DX_SEP_HOST_ECC_PRIVKEYSIZE_OFFSET]   = PrivKeySizeInBytes;        
    Host_messageParam[DX_SEP_HOST_ECC_HASH_OFFSET]          = HashMode;        
	
	
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    &Host_sramOffset,
                                    DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   
   /* send data in buffer */
   Host_maxLength = ((DataInSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)DataIn_ptr , 
                                         DataInSize, 
                                         Host_maxLength,
                                         &Host_sramOffset , 
                                         DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
  
                           
   /* send public key */
   Host_maxLength = ((PublKeySizeInBytes + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PublKeyIn_ptr , 
                                         PublKeySizeInBytes , 
                                         Host_maxLength,
                                         &Host_sramOffset , 
                                         DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send private key */
   Host_maxLength = ((PrivKeySizeInBytes + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PrivKeyIn_ptr, 
                                                     PrivKeySizeInBytes , 
                                                     Host_maxLength,
                                                     &Host_sramOffset , 
                                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* end message */
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* wait for the response */
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   /*-------------------------------------
      start reading message from the SEP 
   --------------------------------------*/
   
   /* check message status from SEP */
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;   
   
}/* END OF SEPQA_ECC_ECDSA_NonIntegratedSignVerify */

/***********************************************************************
* Function Name:                                                           
*  SEPQA_ECC_ECDSA_IntegratedSignVerify         
* 
* Inputs:
* KeyGenType - key gen type (generate internally or build keys)
* DomainID - domain id
* PublKeyIn_ptr   - The pointer to external public key data.
* PublKeySizeInBytes -	Size of public key data in bytes (according to EC domain),
*						it should be equal to 2*modulusSize (CRYS_ECPKI_DomainID_WMDRM10) or 
*						2*modulusSize + 1byte (other domains).
* PrivKeyIn_ptr -      Pointer to private key data. 
* PrivKeySizeInBytes - Size of private key data in bytes. Must be great than null and
*					   less or equall to EC OrderSizeInBytes.
* DataIn_ptr - input data
* DataInSize - input data size
* SignatureSize - size of the sugnature
* HashMode - hash mode
* SignatureSize - the size of signature buffer 
*				  must be not less than 2*OrderSizeInBytes.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                              returns the name of the function that failures
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
*  This function is used to operate the qa external application on the SEP, 
*  that executes Acceptance test for ECC ESDSA Sign & Verify
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with all non buffer parameters of the API
* 3. send all buffers params                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
**************************************************************************************/                                
CRYSError_t SEPQA_ECC_ECDSA_IntegratedSignVerify(
					DxUint32_t			      KeyGenType,
					CRYS_ECPKI_DomainID_t     DomainID,
					DxUint8_t			     *PublKeyIn_ptr,       /*in*/									
					DxUint32_t                PublKeySizeInBytes,  /*in*/
					DxUint8_t			     *PrivKeyIn_ptr,     /*in*/
					DxUint32_t                PrivKeySizeInBytes,/*in*/
					CRYS_ECPKI_HASH_OpMode_t  HashMode,
					DxUint8_t				 *DataIn_ptr,
					DxUint16_t                DataInSize,
					DxUint32_t                SignatureSize,
					char                     *FailureFunctionNameOut_ptr,
					DxUint32_t                FuncNameMaxSize)
{
    /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  
   
   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = 7;

   /* parameters */
   DxUint32_t   Host_messageParam[7] = {0};
   
   /* op code */
   DxUint32_t   Host_opCode = 0;
   	 
  
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   

   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_ECC_ECDSA_IntegratedSignVerify;

   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
   	Host_messageParam[DX_SEP_HOST_ECC_DATASIZE_IN_OFFSET]   = DataInSize;
    Host_messageParam[DX_SEP_HOST_ECC_SIGNATURESIZE_OFFSET] = SignatureSize;
    Host_messageParam[DX_SEP_HOST_ECC_KEYGENTYPE_OFFSET]    = KeyGenType;
    Host_messageParam[DX_SEP_HOST_ECC_DOMAINID_OFFSET]      = DomainID;        
    Host_messageParam[DX_SEP_HOST_ECC_PUBLICKEYSIZE_OFFSET] = PublKeySizeInBytes;        
    Host_messageParam[DX_SEP_HOST_ECC_PRIVKEYSIZE_OFFSET]   = PrivKeySizeInBytes;        
    Host_messageParam[DX_SEP_HOST_ECC_HASH_OFFSET]          = HashMode;        
	
	
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    &Host_sramOffset,
                                    DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   
   /* send data in buffer */
   Host_maxLength = ((DataInSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)DataIn_ptr , 
                                         DataInSize, 
                                         Host_maxLength,
                                         &Host_sramOffset , 
                                         DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
  
                           
   /* send public key */
   Host_maxLength = ((PublKeySizeInBytes + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PublKeyIn_ptr , 
                                         PublKeySizeInBytes , 
                                         Host_maxLength,
                                         &Host_sramOffset , 
                                         DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send private key */
   Host_maxLength = ((PrivKeySizeInBytes + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PrivKeyIn_ptr, 
                                                     PrivKeySizeInBytes , 
                                                     Host_maxLength,
                                                     &Host_sramOffset , 
                                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* end message */
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* wait for the response */
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   /*-------------------------------------
      start reading message from the SEP 
   --------------------------------------*/
   
   /* check message status from SEP */
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;       
}


/***********************************************************************
* Function Name:                                                           
*  SEPQA_ECC_DH         
* 
* Inputs:
* KeyGenType - key gen type (generate internally or build keys)
* DomainID - domain id
* PublKeyIn1_ptr   - The pointer to external public key data for user1.
* PublKeySizeInBytes1 -	Size of public key data in bytes 
* PublKeyIn2_ptr   - The pointer to external public key data for user2.
* PublKeySizeInBytes2 -	Size of public key data in bytes 
* PrivKeyIn1_ptr -      Pointer to private key data for user1. 
* PrivKeySizeInBytes1 - Size of private key data in bytes. Must be great than null and
*					   less or equall to EC OrderSizeInBytes.
* PrivKeyIn2_ptr -      Pointer to private key data for user2. 
* PrivKeySizeInBytes2 - Size of private key data in bytes. Must be great than null and
*					   less or equall to EC OrderSizeInBytes.
* MaxSecretKeySize    = maximal size of the shared secret key
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                              returns the name of the function that failures
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
*  This function is used to operate the qa external application on the SEP, 
*  that executes Diffie-Hellman key shared secret generation
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with all non buffer parameters of the API
* 3. send all buffers params                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
**************************************************************************************/                                
CRYSError_t SEPQA_ECC_DH(
				DxUint32_t			      KeyGenType,
				CRYS_ECPKI_DomainID_t     DomainID,
				DxUint8_t			     *PublKeyIn1_ptr,       /*in*/									
				DxUint32_t                PublKeySizeInBytes1,  /*in*/
				DxUint8_t			     *PublKeyIn2_ptr,       /*in*/									
				DxUint32_t                PublKeySizeInBytes2,  /*in*/
				DxUint8_t			     *PrivKeyIn1_ptr,     /*in*/
				DxUint32_t                PrivKeySizeInBytes1,/*in*/
				DxUint8_t			     *PrivKeyIn2_ptr,     /*in*/
				DxUint32_t                PrivKeySizeInBytes2,/*in*/
				DxUint32_t                MaxSecretKeySize,
				char                     *FailureFunctionNameOut_ptr,
				DxUint32_t                FuncNameMaxSize)
{
    /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  
   
   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = 7;

   /* parameters */
   DxUint32_t   Host_messageParam[7] = {0};
   
   /* op code */
   DxUint32_t   Host_opCode = 0;
   	 
  
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   

   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_ECC_DH;

   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
   	Host_messageParam[DX_SEP_HOST_ECC_DH_KEYGENTYPE_OFFSET]       = KeyGenType;
   	Host_messageParam[DX_SEP_HOST_ECC_DH_DOMAINID_OFFSET]         = DomainID;
    Host_messageParam[DX_SEP_HOST_ECC_DH_PUBLICKEYSIZE1_OFFSET]   = PublKeySizeInBytes1;
    Host_messageParam[DX_SEP_HOST_ECC_DH_PRIVKEYSIZE1_OFFSET]     = PrivKeySizeInBytes1;        
    Host_messageParam[DX_SEP_HOST_ECC_DH_PUBLICKEYSIZE2_OFFSET]   = PublKeySizeInBytes2;        
    Host_messageParam[DX_SEP_HOST_ECC_DH_PRIVKEYSIZE2_OFFSET]     = PrivKeySizeInBytes2;        
    Host_messageParam[DX_SEP_HOST_ECC_DH_MaxSecretKeySize_OFFSET] = MaxSecretKeySize;        
	
	
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    &Host_sramOffset,
                                    DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   
   /* send public key 1 */
   Host_maxLength = ((PublKeySizeInBytes1 + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PublKeyIn1_ptr , 
                                         PublKeySizeInBytes1, 
                                         Host_maxLength,
                                         &Host_sramOffset , 
                                         DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
  
                           
   /* send private key 1*/
   Host_maxLength = ((PrivKeySizeInBytes1 + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PrivKeyIn1_ptr , 
                                         PrivKeySizeInBytes1 , 
                                         Host_maxLength,
                                         &Host_sramOffset , 
                                         DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send public key 2 */
   Host_maxLength = ((PublKeySizeInBytes2 + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PublKeyIn2_ptr , 
                                         PublKeySizeInBytes2, 
                                         Host_maxLength,
                                         &Host_sramOffset , 
                                         DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
  
                           
   /* send private key 2*/
   Host_maxLength = ((PrivKeySizeInBytes2 + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PrivKeyIn2_ptr , 
                                         PrivKeySizeInBytes2, 
                                         Host_maxLength,
                                         &Host_sramOffset , 
                                         DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }

   
   /* end message */
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* wait for the response */
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   /*-------------------------------------
      start reading message from the SEP 
   --------------------------------------*/
   
   /* check message status from SEP */
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;       
}

/***********************************************************************
* Function Name:                                                           
*  SEPQA_ECC_ElGamal_EncDec         
* 
* Inputs:
* KeyGenType - key gen type (generate internally or build keys)
* DomainID - domain id
* PublKeyIn_ptr   - The pointer to external public key data.
* PublKeySizeInBytes -	Size of public key data in bytes (according to EC domain),
*						it should be equal to 2*modulusSize (CRYS_ECPKI_DomainID_WMDRM10) or 
*						2*modulusSize + 1byte (other domains).
* PrivKeyIn_ptr -      Pointer to private key data. 
* PrivKeySizeInBytes - Size of private key data in bytes. Must be great than null and
*					   less or equall to EC OrderSizeInBytes.
* DataIn_ptr - input data
* DataInSize - input data size
* EncryptedMsgSize - size of the encrypted message
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                              returns the name of the function that failures
* FuncNameMaxSize - function name size
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
*  This function is used to operate the qa external application on the SEP, 
*  that executes Diffie-Hellman key shared secret generation
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with all non buffer parameters of the API
* 3. send all buffers params                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
**************************************************************************************/                                
CRYSError_t SEPQA_ECC_ElGamal_EncDec(
						 DxUint32_t			      KeyGenType,
						 CRYS_ECPKI_DomainID_t    DomainID,
						 DxUint8_t			     *PublKeyIn_ptr,       
						 DxUint32_t               PublKeySizeInBytes,  
						 DxUint8_t			     *PrivKeyIn_ptr,     
						 DxUint32_t               PrivKeySizeInBytes,
					     DxUint8_t				 *DataIn_ptr,
						 DxUint16_t               DataInSize,
						 DxUint32_t               EncryptedMsgSize,
						 char                    *FailureFunctionNameOut_ptr,
				         DxUint32_t               FuncNameMaxSize)
{
    /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  
   
   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = 7;

   /* parameters */
   DxUint32_t   Host_messageParam[7] = {0};
   
   /* op code */
   DxUint32_t   Host_opCode = 0;
   	 
  
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   

   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_ECC_ElGamal_EncDec;

   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
   	Host_messageParam[DX_SEP_HOST_ECC_DATASIZE_IN_OFFSET]       = DataInSize;
   	Host_messageParam[DX_SEP_HOST_ECC_SIGNATURESIZE_OFFSET]     = EncryptedMsgSize;
    Host_messageParam[DX_SEP_HOST_ECC_KEYGENTYPE_OFFSET]        = KeyGenType;
    Host_messageParam[DX_SEP_HOST_ECC_DOMAINID_OFFSET]          = DomainID;        
    Host_messageParam[DX_SEP_HOST_ECC_PUBLICKEYSIZE_OFFSET]     = PublKeySizeInBytes;        
    Host_messageParam[DX_SEP_HOST_ECC_PRIVKEYSIZE_OFFSET]       = PrivKeySizeInBytes;        
    Host_messageParam[DX_SEP_HOST_ECC_SIGNATURESIZE_OFFSET]     = EncryptedMsgSize;        
    Host_messageParam[DX_SEP_HOST_ECC_HASH_OFFSET]     = 0;        
	
	
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    &Host_sramOffset,
                                    DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   
   /* send data input buffer */
   Host_maxLength = ((DataInSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)DataIn_ptr , 
                                         DataInSize, 
                                         Host_maxLength,
                                         &Host_sramOffset , 
                                         DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
  
                           
   /* send public key */
   Host_maxLength = ((PublKeySizeInBytes + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PublKeyIn_ptr , 
                                         PublKeySizeInBytes , 
                                         Host_maxLength,
                                         &Host_sramOffset , 
                                         DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send private key */
   Host_maxLength = ((PrivKeySizeInBytes + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PrivKeyIn_ptr , 
                                         PrivKeySizeInBytes, 
                                         Host_maxLength,
                                         &Host_sramOffset , 
                                         DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
  
   
   /* end message */
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* wait for the response */
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   /*-------------------------------------
      start reading message from the SEP 
   --------------------------------------*/
   
   /* check message status from SEP */
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;       
}


/***********************************************************************
* Function Name:                                                           
*  SEPQA_ECC_ImportExportKeyPair         
* 
* Inputs:
* KeyGenType - key gen type (generate internally or build keys)
* DomainID - domain id
* PublKeyIn_ptr   - The pointer to external public key data.
* PublKeySizeInBytes -	Size of public key data in bytes (according to EC domain),
*						it should be equal to 2*modulusSize (CRYS_ECPKI_DomainID_WMDRM10) or 
*						2*modulusSize + 1byte (other domains).
* PrivKeyIn_ptr -      Pointer to private key data. 
* PrivKeySizeInBytes - Size of private key data in bytes. Must be great than null and
*					   less or equall to EC OrderSizeInBytes.
* DataIn_ptr - input data
* DataInSize - input data size
* SignatureSize - size of the sugnature
* HashMode - hash mode
* SignatureSize - the size of signature buffer 
*				  must be not less than 2*OrderSizeInBytes.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                              returns the name of the function that failures
* FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
*  This function is used to operate the qa external application on the SEP, 
*  that executes Acceptance test import/export of the keys
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with all non buffer parameters of the API
* 3. send all buffers params                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
*
**************************************************************************/                                
EXPORT_C DxUint32 SEPQA_ECC_ImportExportKeyPair(
                            	DxUint32_t			      KeyGenType,
                            	CRYS_ECPKI_DomainID_t     DomainID,
                            	DxUint8_t			     *PublKeyIn_ptr,       /*in*/									
                            	DxUint32_t                PublKeySizeInBytes,  /*in*/
                            	DxUint8_t			     *PrivKeyIn_ptr,     /*in*/
                            	DxUint32_t                PrivKeySizeInBytes,/*in*/
                            	CRYS_ECPKI_HASH_OpMode_t  HashMode,
                            	DxUint8_t				 *DataIn_ptr,
                            	DxUint32_t                DataInSize,
                            	DxUint32_t                SignatureSize,
                            	char                     *FailureFunctionNameOut_ptr,
                                DxUint32_t                FuncNameMaxSize)
{
   /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  
   
   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = 7;

   /* parameters */
   DxUint32_t   Host_messageParam[7] = {0};
   
   /* op code */
   DxUint32_t   Host_opCode = 0;
   	 
  
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   

   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_ECC_ImportExportKeyPair;

   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
   	Host_messageParam[DX_SEP_HOST_ECC_DATASIZE_IN_OFFSET]   = DataInSize;
    Host_messageParam[DX_SEP_HOST_ECC_SIGNATURESIZE_OFFSET] = SignatureSize;
    Host_messageParam[DX_SEP_HOST_ECC_KEYGENTYPE_OFFSET]    = KeyGenType;
    Host_messageParam[DX_SEP_HOST_ECC_DOMAINID_OFFSET]      = DomainID;        
    Host_messageParam[DX_SEP_HOST_ECC_PUBLICKEYSIZE_OFFSET] = PublKeySizeInBytes;        
    Host_messageParam[DX_SEP_HOST_ECC_PRIVKEYSIZE_OFFSET]   = PrivKeySizeInBytes;        
    Host_messageParam[DX_SEP_HOST_ECC_HASH_OFFSET]          = HashMode;        
	
	
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    &Host_sramOffset,
                                    DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   
   /* send data in buffer */
   Host_maxLength = ((DataInSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)DataIn_ptr , 
                                         DataInSize, 
                                         Host_maxLength,
                                         &Host_sramOffset , 
                                         DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
  
                           
   /* send public key */
   Host_maxLength = ((PublKeySizeInBytes + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PublKeyIn_ptr , 
                                         PublKeySizeInBytes , 
                                         Host_maxLength,
                                         &Host_sramOffset , 
                                         DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send private key */
   Host_maxLength = ((PrivKeySizeInBytes + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PrivKeyIn_ptr, 
                                                     PrivKeySizeInBytes , 
                                                     Host_maxLength,
                                                     &Host_sramOffset , 
                                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* end message */
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* wait for the response */
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   /*-------------------------------------
      start reading message from the SEP 
   --------------------------------------*/
   
   /* check message status from SEP */
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;   
   
}/* END OF SEPQA_ECC_ImportExportKeyPair */




